from typing import Dict, Any, List
import re

from ..action_space import FormalizationAction, ActionType
from ..symbol_manager import SymbolManager
import core.agent_prompt as AgentPrompt
from utils.json_utils import extract_json
from utils.logger import Logger
from llm.llm_wrapper import LLMWrapper
from llm.message import (
    Message,
    MessageContent,
    ROLE_SYSTEM,
    ROLE_USER,
    TYPE_SETTING,
    TYPE_CONTENT,
)

class StrategicDecompAction(FormalizationAction):
    
    def __init__(self, logger: Logger, llm: LLMWrapper, symbol_manager: SymbolManager):
        super().__init__(logger, llm, symbol_manager)

    def get_type(self):
        return ActionType.STRATEGIC_DECOMPOSITION
    
    def should_apply(self, text: str, context=None) -> bool:
        return True
        
    def apply(self, text: str, context=None) -> Dict[str, Any]:
        try:
            transformed_info = self._analyze_and_decompose(text)
            if not transformed_info:
                return {
                    'success': False,
                    'message': 'Failed to decompose instruction strategically'
                }
            
            steps = transformed_info['steps']

            response, transformed_text = self._attack_target_llm(
                steps
            )
            
            return {
                "success": True,
                "transformed_info": transformed_info,
                "transformed_text": transformed_text,
                "response": response,
            }
        
        except Exception as e:
            self.logger.log_exception(e)
            return {
                'success': False,
                'error': str(e)
            }
        
    def _attack_target_llm(self, steps):

        transformed_text = ""
        responses = ""
        for step in steps:
            step_num = step['step_number']
            step_query = step['instruction']
            system_prompt, user_prompt = AgentPrompt.step_attack_prompt(step_query)
            messages = [
                Message(ROLE_SYSTEM, [MessageContent(TYPE_SETTING, system_prompt)]),
                Message(ROLE_USER, [MessageContent(TYPE_CONTENT, user_prompt)]),
            ]
            transformed_text += f"Step {step_num}: {step_query}\n"
            response = self.llm.generate(messages)
            self.logger.info(f"{self.get_type()} action attack response:\n {response} in step {step_num}")
            responses += f"{step_num}: {response}\n"

        return responses, transformed_text
        
    def _llm_should_apply_strategic_decomp(self, text: str) -> bool:
        try:
            prompt = AgentPrompt.should_strategic_decomp_prompt(text)
            messages = [Message(ROLE_USER, [MessageContent(TYPE_CONTENT, prompt)])]
            response = self.llm.generate(messages).strip().upper()
            self.logger.info(f"Strategic decomposition check for [{text}] - Result: [{response}]")
            return "YES" in response
        except Exception as e:
            self.logger.log_exception(e)
            return False
        
    def _analyze_and_decompose(self, text: str) -> Dict:
        try:
            prompt = AgentPrompt.analyze_strategic_decomp_prompt(text)
            messages = [Message(ROLE_USER, [MessageContent(TYPE_CONTENT, prompt)])]
            response = self.llm.generate(messages)
            self.logger.info(f"Strategic decomposition analysis response: {response}")
            
            data = extract_json(response)
            
            if (isinstance(data, dict) and 
                'steps' in data and 
                isinstance(data['steps'], list) and
                len(data['steps']) > 1):
                return data
            
            return None
            
        except Exception as e:
            self.logger.log_exception(e)
            return None
